import csv
from datetime import datetime, timedelta
import os
import requests

def parse_csv(csv_file):
    flights = []
    with open(csv_file, newline='', encoding='utf-8') as file:
        reader = csv.DictReader(file, delimiter=';')
        for row in reader:
            flight = {
                'Carrier': row.get('Carrier', ''),
                'From': row.get('From', ''),
                'To': row.get('To', ''),
                'Flight number': row.get('Flight number', ''),
                'Departure time': row.get('Departure time', ''),
                'Arrival time': row.get('Arrival time', ''),
                'Booking reference': row.get('Booking reference', ''),
                'Duration': row.get('Duration', ''),
                'Date': row.get('Date', ''),
                'E-ticket': row.get('e-ticket', '')
            }
            flights.append(flight)
    return flights

def get_city_name(iata_code):
    url = f"https://airport-info.p.rapidapi.com/airport?iata={iata_code}"
    headers = {
        'x-rapidapi-host': "airport-info.p.rapidapi.com",
        'x-rapidapi-key': "your-api-key-here"
    }
    response = requests.request("GET", url, headers=headers)
    data = response.json()
    print (data)
    city = data['name']
    return city


def create_ical_entry(flight):
    from_city = get_city_name(flight['From'])
    to_city = get_city_name(flight['To'])

    ical_data = f"BEGIN:VCALENDAR\n"
    
    # Set start and end date/time
    date = flight['Date']
    departure_time = flight['Departure time']
    arrival_time = flight['Arrival time']
    
    start_datetime = datetime.strptime(f"{date} {departure_time}", "%d/%m/%Y %H:%M")
    end_datetime = start_datetime + timedelta(hours=int(flight['Duration'].split(':')[0]), minutes=int(flight['Duration'].split(':')[1]))
    
    ical_data += f"BEGIN:VEVENT\n"
    ical_data += f"UID:{flight['Flight number']}\n"
    ical_data += f"DTSTART:{start_datetime.strftime('%Y%m%dT%H%M%S')}\n"
    ical_data += f"DTEND:{end_datetime.strftime('%Y%m%dT%H%M%S')}\n"
    ical_data += f"SUMMARY:Flight N° {flight['Carrier']}{flight['Flight number']} from {flight['From']} to {flight['To']} ref: {flight['Booking reference']} \n"
    ical_data += f"DESCRIPTION:Please note that the schedules of this booking are local times. Your calendar application automatically adjusts those hours to the time zone schedule.\\n" \
                  "------------------------------------------------------------------------------------------------------\\n" \
                  f"Route: {from_city} {date}\\n" \
                  f"Carrier: {flight['Carrier']}{flight['Flight number']}\\n" \
                  f"Departure: {from_city} {flight['From']} ({date} {departure_time})\\n" \
                  f"Arrival: {to_city} {flight['To']} ({date} {arrival_time})\\n" \
                  f"PNR number: {flight['Booking reference']}\\n" \
                  f"Reservation code: {flight['Booking reference']}\\n" \
                  "Traveler: Bogdan BOTEZATU\\n" \
                  "Arranger: Bogdan BOTEZATU\\n" \
                  f"Duration: {flight['Duration']}\\n" \
				  f"Electronic ticket issued: {flight['E-ticket']}\\n" \
                  "------------------------------------------------------------------------------------------------------\\n" \
                  "We wish you a nice trip!\\n"
    ical_data += f"LOCATION:{from_city} {flight['From']} {to_city} {flight['To']}\n"
    ical_data += f"END:VEVENT\n"
    
    ical_data += f"END:VCALENDAR\n"
    
    return ical_data

def main():
    csv_file = 'flights.csv'  # Change this to your CSV file name
    flights = parse_csv(csv_file)
    for flight in flights:
        departure_datetime = datetime.strptime(flight['Date'], "%d/%m/%Y").strftime("%A, %d. %B %Y")
        file_name = f"Bogdan BOTEZATU {departure_datetime}_{flight['To']}_{flight['Booking reference']}.ics"
        ical_data = create_ical_entry(flight)
        
        with open(file_name, 'w') as file:
            file.write(ical_data)

if __name__ == "__main__":
    main()
